﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using System.IO;
using System.Data.Common;
using System.Reflection;


/*  LINQ to Dataset: Query clauses and keywords 
    
    This examples corresponds to the one for LINQ to Objects.
    Data are now comimg from a real database in disk accessed through ADO.NET.
 */
namespace Lessons
{
    public partial class MainForm : Form
    {
        public MainForm()
        {
            InitializeComponent();
        }


        /* static fields */
        static private readonly string connectionString = @"";
        static private DbProviderFactory factory = null;


        /* static constructor */
        static MainForm()
        {
            string S = Path.GetFullPath(@"..\..\..\LINQ.MDB");
            connectionString = string.Format(@"Provider=Microsoft.Jet.OLEDB.4.0;Data Source=""{0}"";User Id=admin;Password=;", S);
            factory = DbProviderFactories.GetFactory("System.Data.OleDb");
        }


        /* a static method which executes SQL and returns a DataTable object */
        static DataTable Select(string SQL)
        {
            DataTable Result = null;

            if (factory != null)
            {
                DbConnection con = factory.CreateConnection();
                con.ConnectionString = connectionString;

                try
                {
                    con.Open();

                    using (DbCommand cmd = con.CreateCommand())
                    {
                        using (DbDataAdapter adapter = factory.CreateDataAdapter())
                        {
                            cmd.CommandText = SQL;
                            adapter.SelectCommand = cmd;
                            Result = new DataTable();
                            adapter.Fill(Result);
                        }
                    }
                }
                finally
                {
                    if (con.State == ConnectionState.Open)
                        con.Close();
                }

            }

            return Result;
        }


        /* instance fields */
        DataTable tblCity = Select("select * from CITY");
        DataTable tblCountry = Select("select * from COUNTRY");
        DataTable tblCustomer = Select("select * from CUSTOMER");



        /* q is IEnumerable<DataRow> 
           city (range variable) is a DataRow object
           Field<T> is a static generic method are defined at DataRowExtensions class as an extension to DataRow class
           AsEnumerable() and CopyToDataTable() static methods are defined at DataTableExtensions as extensions to DataTable class 
         */
        private void button1_Click(object sender, EventArgs e)
        {
            var q = from city in tblCity.AsEnumerable()
                    where city.Field<string>("Name").StartsWith("A") || city.Field<string>("Name").StartsWith("B")
                    orderby city.Field<string>("Name") descending
                    select city;

            Grid.DataSource = q.CopyToDataTable();
        }



        /* sub-query based on a result list coming from the first query 
           Each from clause can be thought of as a separated foreach statement 
         
           q is IEnumerable<DataRow> */
        private void button2_Click(object sender, EventArgs e)
        {
            var q = from country in tblCountry.AsEnumerable()
                    where (country.Field<string>("Code") == "GR") || (country.Field<string>("Code") == "DK")
                        from customer in tblCustomer.AsEnumerable()
                        where (customer.Field<string>("Country") == country.Field<string>("Code")) && (customer.Field<double>("Debit") > 40)
                        orderby customer.Field<string>("Name") 
                        select customer;

            Grid.DataSource = q.CopyToDataTable();
        }



        /* sub-query.
           the above example split into two steps and uses two distinct query variables.
           The second query uses the IEnumerable<T>.Contains() extension method          
                 
           q is IEnumerable<string>
           q2 is IEnumerable<DataRow>
           */
        private void button3_Click(object sender, EventArgs e)
        {
            var q = from country in tblCountry.AsEnumerable()
                    where (country.Field<string>("Code") == "GR") || (country.Field<string>("Code") == "DK")
                    select country.Field<string>("Code");

            var q2 = from customer in tblCustomer.AsEnumerable()
                     where q.Contains(customer.Field<string>("Country")) && (customer.Field<double>("Debit") > 40)
                     orderby customer.Field<string>("Name")
                     select customer;

            Grid.DataSource = q2.CopyToDataTable();
        }




        /* join 
           
            A LINQ join joins two sequences in an equii-join based on some equality condition.
            Joins in LINQ are always inner (equijoins) joins.

            q is IEnumerable<a> where a is an anonymous type of new { string Name, string Country }

            NOTE: The DataTableExtensions.CopyToDataTable<T>() extension method is a generic with a constraint
            for DataRow class. Meaning that it can be used with DataRow objects only.
            However in the next example the element type is an anonymous type.
            For this and similar example to work a solution is provided taken from
                http://msdn.microsoft.com/en-us/library/bb669096.aspx 
            which adds another overload to CopyToDataTable() without any type constraint.       
         */
        private void button4_Click(object sender, EventArgs e)
        {

            var q = from customer in tblCustomer.AsEnumerable()
                    join country in tblCountry.AsEnumerable() on customer.Field<string>("Country") equals country.Field<string>("Code")
                    orderby country.Field<string>("Code")
                    select new { Name = customer.Field<string>("Name"), Country = country.Field<string>("Name") };

            Grid.DataSource = q.CopyToDataTable();
        }



        /* join 
           
            A LINQ join joins two sequences in an equii-join based on some equality condition.
            Joins in LINQ are always inner (equijoins) joins.

            In this version the result is the same as above, although datasources
            are used here in reverse order 
                 
            q is IEnumerable<a> where a is an anonymous type of new { string Name, string Country }  */
        private void button5_Click(object sender, EventArgs e)
        {
            var q = from country in tblCountry.AsEnumerable()
                    join customer in tblCustomer.AsEnumerable() on country.Field<string>("Code") equals customer.Field<string>("Country")
                    orderby country.Field<string>("Code")
                    select new { Name = customer.Field<string>("Name"), Country = country.Field<string>("Name") };

            Grid.DataSource = q.CopyToDataTable();
        }



        /* helper method: creates and returns a DataTable based on TableName and Rows */
        DataTable ConstructTable(string TableName, IEnumerable<DataRow> Rows)
        {
            DataTable Result;
            if (Rows.Count() > 0)
                Result = Rows.CopyToDataTable();
            else
                Result = new DataTable();

            Result.TableName = TableName;
            return Result;
        }



        /* helper method: returns a DataSet after placing all tables into its Tables  
           and TableNames into cboTables.Items */
        DataSet ConstructDataset(IEnumerable<DataTable> tables)
        {
            DataSet Result = new DataSet();
            cboTables.Items.Clear();

            foreach (DataTable table in tables)
            {
                Result.Tables.Add(table);
                cboTables.Items.Add(table.TableName);
            }

            cboTables.SelectedIndex = 0;
            return Result;
        }


        /* handles the item change of the cboTables */
        private void cboTables_SelectedValueChanged(object sender, EventArgs e)
        {
            DataTable table = Grid.DataSource as DataTable;

            if ((table != null) && (table.DataSet != null))
                Grid.DataSource = table.DataSet.Tables[cboTables.Text];
        }




        /*  group join. 
            
            A LINQ join joins two sequences in an equii-join based on some equality condition.
            Joins in LINQ are always inner (equijoins) joins.
           
            The use of keyword into has a special effect when used with LINQ joins.

            For each element of the left sequence it creates an IEnumerable<T> list, where T 
            is the element type of the right sequence. That list contains those elements of the 
            right sequence which pass the equality condition. This is grouping actually.

            So joins using the keyword into are called group joins.
            There is no SQL equivalent for LINQ group join.     

            In general the keyword into is used to create an identifier (which is another implicitly typed local variable)
            which refers to the results of a group, join or select clause. 
            That identifier can then be used further as a datasource for additional query commands.

            customersPerCountry is IEnumerable<DataRow>
            q is IEnumerable<DataTable> where each table in the sequence is constructed by a call to ConstructTable() method.
            The user may select another table using the cboTables combo box
         
            This query creates a IEnumerable<DataRow> for each group and then calls the ConstructTable()
            in order to create a DataTable for the group           
         */
        private void button6_Click(object sender, EventArgs e)
        {
            var q = from country in tblCountry.AsEnumerable()
                    join customer in tblCustomer.AsEnumerable() on country.Field<string>("Code") equals customer.Field<string>("Country") 
                        into customersPerCountry
                    orderby country.Field<string>("Code")
                    select ConstructTable(country.Field<string>("Name"), customersPerCountry);

            Grid.DataSource = ConstructDataset(q).Tables[0];
        }



        /* join
           simulating a left outer join 
         
           The left join simulation is done by using a second iteration (from clause)
           over the sequence produced by the group join and referenced by the keyword into,
           and forcing ALL the elements of that produced sequence into the final sequence, 
           by using the DefaultIfEmpty() extension method .  
        
           customersPerCountry is IEnumerable<DataRow>
           q is IEnumerable<a> where a is an anonymous type of new { string Country, string Customer } 
         */
        private void button7_Click(object sender, EventArgs e)
        {
            var q = from country in tblCountry.AsEnumerable()
                    join customer in tblCustomer.AsEnumerable() on country.Field<string>("Code") equals customer.Field<string>("Country")
                        into customersPerCountry
                    orderby country.Field<string>("Code")
                        from v in customersPerCountry.DefaultIfEmpty()
                        select new { Country = country.Field<string>("Name"), Customer = (v == null ? string.Empty : (string)v["Name"]) };
             
            Grid.DataSource = q.CopyToDataTable();
        }

 


        /* group .. by clause 
           
          NOTE: a query body must end with a select or group..by clause 
         
          The result of a group..by clause is actually a 
                IEnumerable<IGrouping<TKey, TElement>>         
          that is a sequence of IGrouping<TKey, TElement> objects.
          Each IGrouping<TKey, TElement> object may contain zero or more elements that match the 
          key condition for the group.      
         
         q is a IEnumerable<DataTable>
         g is a IGrouping<string, DataRow>
         
         This example is slightly different from the corresponding one in the LINQ to Objects examples.
         An into clause is added and the grouping result is passed to the ConstructTable().
         */
        private void button8_Click(object sender, EventArgs e)
        {
            /*  here, the implicitly typed variable q would be defined as 
                IEnumerable<DataTable> q = ...  */
            var q = from customer in tblCustomer.AsEnumerable()
                    orderby customer.Field<string>("Country")
                    group customer by customer.Field<string>("Country") into g
                        select ConstructTable(g.Key, g.ToList());

            Grid.DataSource = ConstructDataset(q).Tables[0];
        }




        /* another group..by example using an into clause.
           
           q is IEnumerable<DataTable>
           g is IGrouping<char, DataRow> */
        private void button9_Click(object sender, EventArgs e)
        {
            var q = from customer in tblCustomer.AsEnumerable()
                    group customer by customer.Field<string>("Name")[0] into g
                    orderby g.Key
                    select ConstructTable(g.Key.ToString(), g.ToList());

            Grid.DataSource = ConstructDataset(q).Tables[0];
        }




        /*  grouping and using aggregate functions (sum)            
                         
            A very frequent need is to group a datasource and then apply some aggregate function. 
            One way to achieve this is to group data using an into clause and then call the appropriate 
            aggregate extension method, possibly using a lambda expression.

            The cc is an IGrouping<string, DataRow> 
            The q is an IEnumerable<a> where a is an anonymous type of new { string Country, double Debit } */
        private void button10_Click(object sender, EventArgs e)
        {
            var q = from customer in tblCustomer.AsEnumerable()
                    group customer by customer.Field<string>("Country") into cc
                    orderby cc.Key
                    select new { Country = cc.Key, Debit = cc.Sum(c => c.Field<double>("Debit")) };

            Grid.DataSource = q.CopyToDataTable();
        }



        /* the keyword let and using aggregate functions (sum)
         
           The let keyword can be used inside a query expression in order to create and initialize additional 
           range variables. Those range variables are considered as constants inside the expression. 
           It may be used in further queries though, if they store a queryable type.      
           
           DebitList is an IEnumerable<double> 
           q is an IEnumerable<a> where a is an anonymous type of new { string Country, double Debit } */
        private void button11_Click(object sender, EventArgs e)
        {
            var q = from country in tblCountry.AsEnumerable()
                    let DebitList = (
                        from customer in tblCustomer.AsEnumerable()
                        where (customer.Field<string>("Country") == country.Field<string>("Code"))
                        select customer.Field<double>("Debit"))
                    orderby country.Field<string>("Code")
                    select new { Country = country.Field<string>("Code"), Debit = DebitList.Sum() };

            Grid.DataSource = q.CopyToDataTable();
        }

 
    }
}











namespace Lessons
{

    /*  taken from:     http://msdn.microsoft.com/en-us/library/bb669096.aspx 
        and             http://blogs.msdn.com/aconrad/archive/2007/09/07/science-project.aspx 
     
        This class provides the code needed to implement overloads of the CopyToDataTable() 
        extension method for the IEnumerable<T> without a generic constraint
        as the other versions, those contained in the DataTableExtensions class, do.
        So these new overloads of the CopyToDataTable() can be used with any object
        returned by a query expression, even anonymous types.
     */
    public class ObjectShredder<T>
    {
        private System.Reflection.FieldInfo[] _fi;
        private System.Reflection.PropertyInfo[] _pi;
        private System.Collections.Generic.Dictionary<string, int> _ordinalMap;
        private System.Type _type;

        // ObjectShredder constructor.
        public ObjectShredder()
        {
            _type = typeof(T);
            _fi = _type.GetFields();
            _pi = _type.GetProperties();
            _ordinalMap = new Dictionary<string, int>();
        }

        /// <summary>
        /// Loads a DataTable from a sequence of objects.
        /// </summary>
        /// <param name="source">The sequence of objects to load into the DataTable.</param>
        /// <param name="table">The input table. The schema of the table must match that 
        /// the type T.  If the table is null, a new table is created with a schema 
        /// created from the public properties and fields of the type T.</param>
        /// <param name="options">Specifies how values from the source sequence will be applied to 
        /// existing rows in the table.</param>
        /// <returns>A DataTable created from the source sequence.</returns>
        public DataTable Shred(IEnumerable<T> source, DataTable table, LoadOption? options)
        {
            // Load the table from the scalar sequence if T is a primitive type.
            if (typeof(T).IsPrimitive)
            {
                return ShredPrimitive(source, table, options);
            }

            // Create a new table if the input table is null.
            if (table == null)
            {
                table = new DataTable(typeof(T).Name);
            }

            // Initialize the ordinal map and extend the table schema based on type T.
            table = ExtendTable(table, typeof(T));

            // Enumerate the source sequence and load the object values into rows.
            table.BeginLoadData();
            using (IEnumerator<T> e = source.GetEnumerator())
            {
                while (e.MoveNext())
                {
                    if (options != null)
                    {
                        table.LoadDataRow(ShredObject(table, e.Current), (LoadOption)options);
                    }
                    else
                    {
                        table.LoadDataRow(ShredObject(table, e.Current), true);
                    }
                }
            }
            table.EndLoadData();

            // Return the table.
            return table;
        }

        public DataTable ShredPrimitive(IEnumerable<T> source, DataTable table, LoadOption? options)
        {
            // Create a new table if the input table is null.
            if (table == null)
            {
                table = new DataTable(typeof(T).Name);
            }

            if (!table.Columns.Contains("Value"))
            {
                table.Columns.Add("Value", typeof(T));
            }

            // Enumerate the source sequence and load the scalar values into rows.
            table.BeginLoadData();
            using (IEnumerator<T> e = source.GetEnumerator())
            {
                Object[] values = new object[table.Columns.Count];
                while (e.MoveNext())
                {
                    values[table.Columns["Value"].Ordinal] = e.Current;

                    if (options != null)
                    {
                        table.LoadDataRow(values, (LoadOption)options);
                    }
                    else
                    {
                        table.LoadDataRow(values, true);
                    }
                }
            }
            table.EndLoadData();

            // Return the table.
            return table;
        }

        public object[] ShredObject(DataTable table, T instance)
        {

            FieldInfo[] fi = _fi;
            PropertyInfo[] pi = _pi;

            if (instance.GetType() != typeof(T))
            {
                // If the instance is derived from T, extend the table schema
                // and get the properties and fields.
                ExtendTable(table, instance.GetType());
                fi = instance.GetType().GetFields();
                pi = instance.GetType().GetProperties();
            }

            // Add the property and field values of the instance to an array.
            Object[] values = new object[table.Columns.Count];
            foreach (FieldInfo f in fi)
            {
                values[_ordinalMap[f.Name]] = f.GetValue(instance);
            }

            foreach (PropertyInfo p in pi)
            {
                values[_ordinalMap[p.Name]] = p.GetValue(instance, null);
            }

            // Return the property and field values of the instance.
            return values;
        }

        public DataTable ExtendTable(DataTable table, Type type)
        {
            // Extend the table schema if the input table was null or if the value 
            // in the sequence is derived from type T.            
            foreach (FieldInfo f in type.GetFields())
            {
                if (!_ordinalMap.ContainsKey(f.Name))
                {
                    // Add the field as a column in the table if it doesn't exist
                    // already.
                    DataColumn dc = table.Columns.Contains(f.Name) ? table.Columns[f.Name]
                        : table.Columns.Add(f.Name, f.FieldType);

                    // Add the field to the ordinal map.
                    _ordinalMap.Add(f.Name, dc.Ordinal);
                }
            }
            foreach (PropertyInfo p in type.GetProperties())
            {
                if (!_ordinalMap.ContainsKey(p.Name))
                {
                    // Add the property as a column in the table if it doesn't exist
                    // already.
                    DataColumn dc = table.Columns.Contains(p.Name) ? table.Columns[p.Name]
                        : table.Columns.Add(p.Name, p.PropertyType);

                    // Add the property to the ordinal map.
                    _ordinalMap.Add(p.Name, dc.Ordinal);
                }
            }

            // Return the table.
            return table;
        }
    }


    /* serves just as a container for the CopyToDataTable() extension methods */
    public static class CustomLINQtoDataSetMethods
    {
        public static DataTable CopyToDataTable<T>(this IEnumerable<T> source)
        {
            if (source is IEnumerable<DataRow>)
                return DataTableExtensions.CopyToDataTable<DataRow>(source as IEnumerable<DataRow>);

            return new ObjectShredder<T>().Shred(source, null, null);
        }

        public static DataTable CopyToDataTable<T>(this IEnumerable<T> source, DataTable table, LoadOption options)
        {
            if (source is IEnumerable<DataRow>)
            {
                 DataTableExtensions.CopyToDataTable<DataRow>(source as IEnumerable<DataRow>, table, options);
                 return table;
            }

            return new ObjectShredder<T>().Shred(source, table, options);
        }

    }

}